/*
 * FlashEdit.c
 *
 *  Created on: Jan 30, 2025
 *      Author: nx827554
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdbool.h>

#include "FlashEdit.h"
#include "main.h"
#include "AppSmarTag.h"
#include "SmarTag2_nfctag.h"
#include "SmartNFC.h"
#include "SmarTag2.h"
#include "TagType5.h"
#include "SmartNFCType.h"

static uint32_t GetBank(uint32_t Addr);
static uint32_t GetPage(uint32_t Addr);
void updatePageNo(uint32_t pageNo);
void updateSampleCounter();
void FlashTest();
HAL_StatusTypeDef ReadMem(uint8_t pageReq);
HAL_StatusTypeDef WriteMem(uint64_t DataBuf64);

// defining area of flash usable for environmental data storage
#define FLASH_READINGS_BASE			(int)((FLASH_BASE +(((FLASH_END-FLASH_BASE)+1)/4)))// page beginning 25% into flash memory
#define FLASH_READINGS_END			FLASH_END
#define MAX_NO_PAGES				(int)((FLASH_READINGS_END-FLASH_READINGS_BASE +1)/FLASH_PAGE_SIZE)
#define MAX_READINGS_PER_PAGE		(int)((FLASH_PAGE_SIZE-48)/NUMBYTES)
#define MAX_NO_READINGS 			MAX_NO_PAGES*MAX_READINGS_PER_PAGE

#define NUMBYTES 					8//size of environmental readings in bytes
#define BUFFTOFLASH_SIZE 			(int)(FLASH_PAGE_SIZE/NUMBYTES)

//buffers/pointers for transporting data
static uint64_t bufToFlash[BUFFTOFLASH_SIZE] = { 0xFFFFFFFF };
int bufToFlashPointer = BUFFTOFLASH_SIZE-1;

static uint64_t	flashToNFC[STSMART_NFC_MAX_SIZE/8] = { 0xFFFFFFFF };
uint32_t flashToNFCPointer = 0;// align pointer with final 32 bits

uint32_t writeToFlashPointer = ((FLASH_READINGS_END+1) - FLASH_PAGE_SIZE);
uint32_t pageNo = 0;//number of pages environmental data has been written to

int memFull =0;//flag for filled flash memory


/**
* @brief  find bank that address belongs to
* @param  uint32_t Addr address in question
*/
static uint32_t GetBank(uint32_t Addr){
	uint32_t bank=0;

	if (READ_BIT(SYSCFG->MEMRMP, SYSCFG_MEMRMP_FB_MODE) == 0){
		//No Bank swap
		if (Addr < (FLASH_BASE + FLASH_BANK_SIZE))
			bank = FLASH_BANK_1;
		else
			bank = FLASH_BANK_2;
	} else {
		//Bank swap
		if (Addr < (FLASH_BASE + FLASH_BANK_SIZE))
			bank = FLASH_BANK_2;
		else
			bank = FLASH_BANK_1;
	}
	return bank;
}

/**
* @brief  find page that address belongs to
* @param  uint32_t Addr address in question
*/
static uint32_t GetPage(uint32_t Addr){
	uint32_t page=0;
	if (Addr < (FLASH_BASE + FLASH_BANK_SIZE))
		page = (Addr - FLASH_BASE) / FLASH_PAGE_SIZE;
	else
		page = (Addr - (FLASH_BASE + FLASH_BANK_SIZE)) / FLASH_PAGE_SIZE;

	return page;
}

/**
* @brief  update page counter in nfc memory
* @param  uint32_t pageNo page number data is currently being written to
*/
void updatePageNo(uint32_t pageNo){
	int ErrorCounter = 0;
	while(1){
		if(BSP_NFCTAG_WriteData( BSP_NFCTAG_INSTANCE, (uint8_t *)&pageNo, LogDefinition.SampleCounterAddress+4, 4)==NFCTAG_OK){
			break;
		}

		if(ErrorCounter >=5){
			STNFC_Error_Handler(STNFC_WRITING_ERROR);
			break;
		}
		ErrorCounter++;
		HAL_Delay(10);
	}

}
/**
* @brief  update sample counter in nfc memory
*/
void updateSampleCounter(){
	int ErrorCounter = 0;
	while(1){
		if(BSP_NFCTAG_WriteData( BSP_NFCTAG_INSTANCE, (uint8_t *)&LogDefinition.SampleCounter, LogDefinition.SampleCounterAddress, 4)==NFCTAG_OK){
			break;
		}

		if(ErrorCounter >=5){
			STNFC_Error_Handler(STNFC_WRITING_ERROR);
			break;
		}
		ErrorCounter++;
		HAL_Delay(10);
	}
}

/**
* @brief  Fill flash memory quickly with simulated results for testing/development
*/
void FlashTest(){
	printf("Start FlashTest\r\n");

	uint64_t DataBuf64;
	uint32_t *DataBuf32 = (uint32_t *)&DataBuf64;

	uint32_t testTime = 0;// equivalent to RTCDDT

	int loopTime = 60; // how much time passes per loop

	// Generate Sensor SampleRates
	struct TestSensors {
		int SampleRate;
		int minLimit;
		int maxLimit;

		float Value;

		int Time;// individual time counter (since last Reading)
		bool Flag;// flag for sensor activation
	};

	struct TestSensors temp;// create artificial Sensors
	struct TestSensors hum;
	struct TestSensors press;

	temp.Time = 0;
	hum.Time = 0;
	press.Time = 0;

	temp.Flag = false;
	hum.Flag = false;
	press.Flag = false;

	temp.SampleRate = 10*60;// set sampleRates
	hum.SampleRate = 15*60;
	press.SampleRate = 20*60;

	temp.minLimit = 5;// set lower and upper limits for simulated data
	temp.maxLimit = 30;
	hum.minLimit = 10;
	hum.maxLimit = 80;
	press.minLimit = 950;
	press.maxLimit = 1050;

	temp.Value = rand() % 10 + 10;// set starting artificial conditions
	hum.Value = rand() % 5 + 20;
	press.Value = rand() % 30 + 1000;


	printf("TEST BEGIN\r\n");
	while(1){
		if(TagStatus == TagScanning && RFActivity2 == 1){// detect RFID usage mid function
			RFActivity2 = 0;
			SmarTagAppDetectRFActivity();
		}
		if(temp.Flag == true || hum.Flag == true || press.Flag == true){
			if(temp.Flag == true){// if samplerate has passed record and save value and time
				temp.Flag = false;
				DataBuf32[0] = SHT40_TEMP_VS_ID |
						(((uint32_t)testTime)<<4);
				DataBuf32[1] = SHT40_TEMP_SAMPLE_TO_CODED(temp.Value);

				if(WriteMem(DataBuf64)!= HAL_OK) printf("ERROR: Writing Temperature\r\n");

				if(memFull == 1){
						printf("Test Complete Memory Full\r\n\t");
						break;
				}
			}

			if(hum.Flag == true){// if samplerate has passed record and save value and time
				hum.Flag = false;
				DataBuf32[0] = SHT40_HUM_VS_ID |
						(((uint32_t)testTime)<<4);
				DataBuf32[1] = SHT40_HUM_SAMPLE_TO_CODED(hum.Value);

				if(WriteMem(DataBuf64)!= HAL_OK) printf("ERROR: Writing Humidity\r\n");

				if(memFull == 1){
						printf("Test Complete Memory Full\r\n\t");
						break;
				}
			}

			if(press.Flag == true){// if samplerate has passed record and save value and time
				press.Flag = false;
				DataBuf32[0] = LPS22DF_VS_ID |
						(((uint32_t)testTime)<<4);
				DataBuf32[1] = LPS22DF_SAMPLE_TO_CODED(press.Value);

				if(WriteMem(DataBuf64)!= HAL_OK) printf("ERROR: Writing Temp\r\n");

				if(memFull == 1){
						printf("Test Complete Memory Full\r\n\t");
						break;
				}
			}
		}
		int change = rand()%5+1;// cause artificial drift in temperature
		if(change <=3)	temp.Value += ((float)change-2)/5;
		if(temp.Value> temp.maxLimit) temp.Value = temp.maxLimit;
		if(temp.Value< temp.minLimit) temp.Value = temp.maxLimit;

		change = rand()%5+1;// cause artificial drift in humidity
		hum.Value += ((float)change-2)/2;
		if(hum.Value> hum.maxLimit) hum.Value = hum.maxLimit;
		if(hum.Value< hum.minLimit) hum.Value = hum.maxLimit;

		change = rand()%5+1;// cause artificial drift in pressure
		press.Value += ((float)change-2)/10;
		if(press.Value> press.maxLimit) press.Value = press.maxLimit;
		if(press.Value< press.minLimit) press.Value = press.maxLimit;


		testTime+=loopTime;// simulate time passing

		temp.Time += loopTime;
		hum.Time += loopTime;
		press.Time += loopTime;

		if(temp.Time >= temp.SampleRate){// flag and reset samplerates for any events
			temp.Flag = true;
			temp.Time -= temp.SampleRate;
		}
		if(hum.Time >= hum.SampleRate){
			hum.Flag = true;
			hum.Time -= hum.SampleRate;
		}
		if(press.Time >= press.SampleRate){
			press.Flag = true;
			press.Time -= press.SampleRate;
		}
	}

	printf("TEST COMPLETE\r\n");
}


/**
* @brief   transfer environmental data from Flash memory to NFC chip
* @return Hal_StatusTypeDef: Return Error status
*/
HAL_StatusTypeDef ReadMem(uint8_t pageReq){
	uint64_t data64;
	uint32_t pageAddress = ((FLASH_READINGS_END+1)-((pageReq)*FLASH_PAGE_SIZE));

	flashToNFCPointer = 0;

	if(pageAddress < writeToFlashPointer){// if page empty, return error
		printf("ERROR: No readings in selection\r\n");
		return HAL_ERROR;
	}
	if(pageReq>pageNo||pageAddress>FLASH_READINGS_END){// if invalid page address requested, return error
		printf("ERROR: Address Out of Bounds\r\n");
		return HAL_ERROR;
	}

	int pagesRead =0;
	for(pagesRead = 0; pagesRead< 2; pagesRead++){// read pages into buffer
		if (pageReq == pageNo){// read from bufToFlash Buffer
			if(pageReq != MAX_NO_PAGES){
				for(int Address = (BUFFTOFLASH_SIZE-1); Address>=0; Address--){//read data from back of page and place into array forwards

					flashToNFC[flashToNFCPointer] =  bufToFlash[Address];
					flashToNFCPointer+=1;
				}
			}
			else{
				for(int Address = pageAddress+(FLASH_PAGE_SIZE-NUMBYTES); Address>=pageAddress; Address-=NUMBYTES){//read data from back of page and place into array forwards
					data64 = *(uint64_t *)Address;// read flash to 64 bit variable
					flashToNFC[flashToNFCPointer] = data64;
					flashToNFCPointer+=1;
				}
			}

		}
		else if(pageReq <pageNo){
			for(int Address = pageAddress+(FLASH_PAGE_SIZE-NUMBYTES); Address>=pageAddress; Address-=NUMBYTES){//read data from back of page and place into array forwards
				data64 = *(uint64_t *)Address;// read flash to 64 bit variable
				flashToNFC[flashToNFCPointer] = data64;
				flashToNFCPointer+=1;
			}
		}
		else{
			printf("\r\n\t ERROR: Page Does Not Exist\r\n");// return error if directed to blank page
			break;
		}
		pageReq+=1;
		pageAddress-=FLASH_PAGE_SIZE;
	}


	// write entire buffer to NFC
 	printf("\r\n\t\t Writing Flash Page to NFC:\r\n");

	int writeAddress = 44;
	int ErrorCount = 0;

	uint32_t dataBuf32 = (uint32_t)((flashToNFC[(writeAddress/8)]>>32)&0xFFFFFFFF);// only write 32 bits to first reading for page header
	while(1){
		if(TagStatus == TagScanning && RFActivity2 == 1){// if RFID status is changing mid function, detect change
			RFActivity2 = 0;
			SmarTagAppDetectRFActivity();
		}

		if(BSP_NFCTAG_WriteData(BSP_NFCTAG_INSTANCE, (uint8_t*)&dataBuf32,writeAddress, 4)==NFCTAG_OK){
			writeAddress+=4;
			break;
		}
		STNFC_Error_Handler(STNFC_WRITING_ERROR);
		if(ErrorCount >=5){
			printf("Writing Error: Break \r\n");
			return HAL_ERROR;
		}
		ErrorCount++;
		HAL_Delay(10);
	}

	for(int i = writeAddress/8; i< flashToNFCPointer;i++){
		//printf("Writing:\t%d\r", i);

		while(1){
			if(TagStatus == TagScanning && RFActivity2 == 1){// detect RFID usage mid function
				RFActivity2 = 0;
				SmarTagAppDetectRFActivity();
			}

			if(BSP_NFCTAG_WriteData(BSP_NFCTAG_INSTANCE, (uint8_t*)&flashToNFC[i],writeAddress, NUMBYTES)==NFCTAG_OK){
				break;
			}
			STNFC_Error_Handler(STNFC_WRITING_ERROR);
			if(ErrorCount >=5){// allow 5 attempts to correct error
				printf("Writing Error: Break \r\n");
				return HAL_ERROR;
			}
			ErrorCount++;
			HAL_Delay(10);
		}

		writeAddress += NUMBYTES;
	}
	BSP_LED_On();
	HAL_Delay(300);
	BSP_LED_Off();
	printf("\r\n\t\t\t FLASH TO NFC COMPLETE\r\n\n\n");
	return HAL_OK;
}

/**
* @brief  write sensorData to memory.
* 			For NFC, write directly to NFC in a 64 word.
* 			For Flash, save to a buffer until one pages worth of readings, then write all to flash.
* @param uint64_t info environmental data to be stored
* @return Hal_StatusTypeDef: Return Error status
*/
HAL_StatusTypeDef WriteMem(uint64_t info){
	uint64_t DataBuf64 = info;
	if(boardMem == NFC){// if memory type is NFC, write to next available space in 64 bit words

		if (BSP_NFCTAG_WriteData(BSP_NFCTAG_INSTANCE, (uint8_t*)&DataBuf64, LogDefinition.LastSamplePointer, 8) != NFCTAG_OK) {
			STNFC_Error_Handler(STNFC_WRITING_ERROR);
			return HAL_ERROR;
		}
		LogDefinition.LastSamplePointer+=8;

		/* Increment the new Sample Counter until the end of the Tag */
		NfcType5_UpdateSampleCounter(&LogDefinition,8);

		/* Update Sample Counter and Last Sample Pointer */
		UpdateLastSamplePointerAndSampleCounter(&LogDefinition);
		if(LogDefinition.LastSamplePointer+8> STSMART_NFC_MAX_SIZE) memFull = 1;
	}

	else if(boardMem == Flash){// if memory type is flash, save to a buffer 1 page long then write entire page to flash
		static FlashStatus_t FlashStatus = Flash_Empty;//declares flashstatus  empty, full, ok, error
		static int initialiseBuffer = 1;
		static uint16_t pageSampleNo = 0;
		if (bufToFlashPointer<0){//if another reading would overflow the buffer, write to flash
			if(FlashStatus<Flash_ERROR){

				FLASH_WaitForLastOperation((uint32_t)0x80);
				HAL_FLASH_Unlock();

				__HAL_FLASH_CLEAR_FLAG(FLASH_FLAG_OPTVERR);

				int i = 0;
				for (int address = writeToFlashPointer; address < (writeToFlashPointer + FLASH_PAGE_SIZE ); address +=8){

					//Have to program using a 64 bit word
					if(HAL_FLASH_Program(FLASH_TYPEPROGRAM_DOUBLEWORD, address, bufToFlash[i]) != HAL_OK){
						HAL_FLASH_Lock();
						FLASH_WaitForLastOperation((uint32_t)0x80);
						printf("\t ERROR: writing Buffer to Flash\r\n");
						FlashStatus = Flash_ERROR;
					}
					i++;
				}
				printf("\t Written to Flash\r\n");
				HAL_FLASH_Lock();
				FLASH_WaitForLastOperation((uint32_t)0x80);

				writeToFlashPointer -= FLASH_PAGE_SIZE;//move pointer to next write position

				for(int i = 0; i< BUFFTOFLASH_SIZE; i++){// reset buffer
					bufToFlash[i] = 0xFF;
				}
				initialiseBuffer = 1;

				if(writeToFlashPointer<FLASH_READINGS_BASE){// if flash has been filled, update flag and print message
					FlashStatus = Flash_Full;
					memFull = 1;
					printf("\t Max Number of Reads saved to FLASH: Continue to save reads to buffer\r\n");
					return HAL_OK;
				}
			}else{
				printf("\t Error with Flash\r\n");
			}
		}
		if(initialiseBuffer == 1 || pageNo == 0){
			bufToFlashPointer = BUFFTOFLASH_SIZE-6;// reset buffer pointer add space for NDEF header
			// Add PageHeader
			pageSampleNo = 0;
			uint8_t *buftoFlashHeader8 = (uint8_t *)&bufToFlash[BUFFTOFLASH_SIZE-6];
			buftoFlashHeader8[4] = 0xFA;// return command (F means return, A means SensorRead)
			buftoFlashHeader8[5] = pageNo;// flash page number
			buftoFlashHeader8[6] =(uint8_t)(pageSampleNo&0xFF);// number of reads included in page a 16 bit number to be updated after every read
			buftoFlashHeader8[7] =(uint8_t)((pageSampleNo>>8)&0xFF);// number of reads included in page a 16 bit number to be updated after every read

			bufToFlashPointer -=1;
			initialiseBuffer =0;
			pageNo ++;
			updatePageNo(pageNo);// page number will always be one greater than that written in buffer to serve as number of pages

		}

		if (bufToFlashPointer>=0){// if the reading won't overflow flash buffer add reading (size of page)
			/*saving reading to buffer*/
			bufToFlash[bufToFlashPointer] = DataBuf64;// add reading to buffer
			bufToFlashPointer -= 1;// start from end of buffer and work way backwards
			LogDefinition.SampleCounter ++;
			updateSampleCounter();
			pageSampleNo++;
			uint8_t *buftoFlashHeader8 = (uint8_t *)&bufToFlash[BUFFTOFLASH_SIZE-6];
			buftoFlashHeader8[4] = 0xFA;// return command (F means return, A means SensorRead)
			buftoFlashHeader8[5] = pageNo;// flash page number
			buftoFlashHeader8[6] =(uint8_t)(pageSampleNo&0xFF);// number of reads included in page a 16 bit number to be updated after every read
			buftoFlashHeader8[7] =(uint8_t)((pageSampleNo>>8)&0xFF);// number of reads included in page a 16 bit number to be updated after every read
//			if(pageSampleNo == 250){
//				memFull =1;
//				return HAL_OK;
//			}
		}
	}

	/* if boardmem set to another type return error*/
	else{
		printf("\r\n\t\tERROR: Memory Type not found\r\n");
		return HAL_ERROR;
	}

	return HAL_OK;
}


/**
* @brief  erase section of flash dedicated to environmental readings
* @return Hal_StatusTypeDef: Return Error status
*/
HAL_StatusTypeDef EraseMem(){

	for(int i =FLASH_READINGS_BASE; i<FLASH_READINGS_END; i+= FLASH_PAGE_SIZE){
		if(TagStatus == TagScanning && RFActivity2 == 1){// detect RFID usage mid function
			RFActivity2 = 0;
			SmarTagAppDetectRFActivity();
		}

		FLASH_WaitForLastOperation((uint32_t)0x80);

		HAL_FLASH_Unlock();
		__HAL_FLASH_CLEAR_FLAG(FLASH_FLAG_OPTVERR);

		//Perform Internal Flash erase
		static FLASH_EraseInitTypeDef EraseInitStruct;
		uint32_t PAGEError;
		EraseInitStruct.TypeErase   = FLASH_TYPEERASE_PAGES;
		EraseInitStruct.Banks       = GetBank(i);
		EraseInitStruct.Page        = GetPage(i);
		EraseInitStruct.NbPages     = 1;
		if (HAL_FLASHEx_Erase(&EraseInitStruct, &PAGEError) != HAL_OK)
		{
		  HAL_FLASH_Lock();
		  printf("\r\n\t ERROR Erasing all Readings");
					return HAL_ERROR;
		}

		HAL_FLASH_Lock();

	}
	if(memFull == 1)memFull = 0;

	return HAL_OK;
}



